import { Head } from '@inertiajs/react';
import { CreditCard, Users, Smartphone, QrCode } from 'lucide-react';
import { ReactNode, useEffect, useState } from 'react';
import { useTranslation } from 'react-i18next';
import { LanguageSwitcher } from '@/components/language-switcher';
import { useBrand } from '@/contexts/BrandContext';
import { useAppearance, THEME_COLORS } from '@/hooks/use-appearance';

interface AuthLayoutProps {
    children: ReactNode;
    title: string;
    description?: string;
    icon?: ReactNode;
    status?: string;
    statusType?: 'success' | 'error';
}

export default function AuthLayout({
    children,
    title,
    description,
    icon,
    status,
    statusType = 'success',
}: AuthLayoutProps) {
    const { t } = useTranslation();
    const [mounted, setMounted] = useState(false);
    const { logoLight, logoDark, themeColor, customColor } = useBrand();
    const { appearance } = useAppearance();
    
    const currentLogo = appearance === 'dark' ? logoDark : logoLight;
    const primaryColor = themeColor === 'custom' ? customColor : THEME_COLORS[themeColor as keyof typeof THEME_COLORS];

    useEffect(() => {
        setMounted(true);
    }, []);

function hexToAdjustedRgba(hex, opacity = 1, adjust = 0) {
    hex = hex.replace("#", "");
    let r = parseInt(hex.slice(0, 2), 16);
    let g = parseInt(hex.slice(2, 4), 16);
    let b = parseInt(hex.slice(4, 6), 16);
    const clamp = (v) => Math.max(-1, Math.min(1, v));
    const getF = (ch) =>
        typeof adjust === "number" ? clamp(adjust) : clamp(adjust[ch] ?? 0);
    const adj = (c, f) =>
        f < 0 ? Math.floor(c * (1 + f)) : Math.floor(c + (255 - c) * f);
    const rr = adj(r, getF("r"));
    const gg = adj(g, getF("g"));
    const bb = adj(b, getF("b"));
    return opacity === 1
        ? `#${rr.toString(16).padStart(2, "0")}${gg
            .toString(16)
            .padStart(2, "0")}${bb.toString(16).padStart(2, "0")}`.toUpperCase()
        : `rgba(${rr}, ${gg}, ${bb}, ${opacity})`;
}

    return (
        <div className="flex min-h-screen w-full bg-slate-50 dark:bg-slate-900">
            <Head title={title} />
            
            {/* Left side - SVG illustration */}
            <div 
                className="hidden lg:block lg:w-1/2 relative overflow-hidden"
                style={{ backgroundColor: primaryColor }}
            >
                {/* Background pattern */}
                <div className="absolute inset-0 opacity-10">
                    <div className="absolute top-0 left-0 w-full h-full">
                        {Array.from({ length: 20 }).map((_, i) => (
                            <div
                                key={i}
                                className="absolute w-2 h-2 bg-white rounded-full animate-pulse"
                                style={{
                                    left: `${Math.random() * 100}%`,
                                    top: `${Math.random() * 100}%`,
                                    animationDelay: `${Math.random() * 3}s`,
                                    animationDuration: `${2 + Math.random() * 2}s`
                                }}
                            />
                        ))}
                    </div>
                </div>
                
                <div 
                    className="absolute inset-0"
                    style={{ backgroundColor: primaryColor }}
                >
                    <div className="h-full flex items-center justify-center p-8 relative z-10">
                        <div className='max-w-[700px] mx-auto w-full'>
                            <svg width="714" height="704" viewBox="0 0 714 704" fill="none" xmlns="http://www.w3.org/2000/svg" className="size-full">
                            <style>
                                    {`
                                    @keyframes dashAnimation {
                                    to {
                                        stroke-dashoffset: -16;
                                    }
                                    }

                                    .animated-border {
                                    animation: dashAnimation 0.7s linear infinite;
                                    }

                                    @keyframes pulse {
                                    0%, 100% {
                                        transform: scale(1);
                                    }
                                    50% {
                                        transform: scale(1.08);
                                    }
                                    }

                                    .pulse {
                                    animation: pulse 2s ease-in-out infinite;
                                    transform-origin: center;
                                    transform-box: fill-box;
                                    }

                                    @keyframes pulse-out {
                                    0%, 100% {
                                        transform: scale(1);
                                    }
                                    50% {
                                        transform: scale(1.08);
                                    }
                                    }

                                    .pulse-out {
                                    animation: pulse-out 2s ease-in-out infinite;
                                    transform-origin: center;
                                    transform-box: fill-box;
                                    }

                                    .pulse-delayed {
                                    animation: pulse-out 3.5s ease-in-out infinite;
                                    transform-origin: center;
                                    transform-box: fill-box;
                                    }

                                    @keyframes pulse-large {
                                    0%, 100% {
                                        transform: scale(1);
                                    }
                                    50% {
                                        transform: scale(1.3);
                                    }
                                    }

                                    .pulse-large {
                                    animation: pulse-large 3s ease-in-out infinite;
                                    transform-origin: center;
                                    transform-box: fill-box;
                                    }
                                `}
                                </style>
                                <rect x="22" y="10" width="670" height="670" rx="60" fill={hexToAdjustedRgba(primaryColor, 1, 0.096)} />
                                <path className="animated-border" d="M54 598V108.41V102C54 74.3858 76.3858 52 104 52H610C637.614 52 660 74.3858 660 102V598C660 625.614 637.614 648 610 648H104C76.3858 648 54 625.614 54 598Z" fill={hexToAdjustedRgba(primaryColor,  1, 0)} stroke="white" strokeWidth="3" strokeDasharray="8 8" />
                                <path className="pulse-large" d="M122.033 317L124.498 322.709L129.066 325.791L124.498 328.873L122.033 334.582L119.567 328.873L115 325.791L119.567 322.709L122.033 317Z" fill="white" fillOpacity="0.15" />
                                <path d="M169.505 363.594L171.354 367.59L174.78 369.748L171.354 371.905L169.505 375.901L167.656 371.905L164.23 369.748L167.656 367.59L169.505 363.594Z" fill="white" fillOpacity="0.15" />
                                <path d="M164.275 502L166.124 505.997L169.549 508.154L166.124 510.311L164.275 514.308L162.426 510.311L159 508.154L162.426 505.997L164.275 502Z" fill="white" fillOpacity="0.15" />
                                <path d="M510.791 400L513.873 406.851L519.582 410.549L513.873 414.248L510.791 421.099L507.709 414.248L502 410.549L507.709 406.851L510.791 400Z" fill="white" fillOpacity="0.15" />
                                <path d="M593.791 554L596.873 560.851L602.582 564.549L596.873 568.248L593.791 575.099L590.709 568.248L585 564.549L590.709 560.851L593.791 554Z" fill="white" fillOpacity="0.15" />
                                <path d="M593.791 110L596.873 116.851L602.582 120.549L596.873 124.248L593.791 131.099L590.709 124.248L585 120.549L590.709 116.851L593.791 110Z" fill="white" fillOpacity="0.15" />
                                <path className="pulse-large" d="M224.791 70L227.873 76.8512L233.582 80.5495L227.873 84.2477L224.791 91.0989L221.709 84.2477L216 80.5495L221.709 76.8512L224.791 70Z" fill="white" fillOpacity="0.15" />
                                <g filter="url(#filter0_d_1944_559)">
                                    <circle cx="356.917" cy="647.917" r="44.9167" fill={hexToAdjustedRgba(primaryColor, 1, { r:  0.1125, g: -0.1141304348, b: -0.0390625 })} />
                                    <circle cx="356.917" cy="647.917" r="44.7667" stroke="white" strokeWidth="0.3" />
                                </g>
                                <g filter="url(#filter1_d_1944_559)">
                                    <circle cx="357" cy="648" r="48.85" stroke="white" strokeWidth="0.3" strokeDasharray="2.6 2.6" shapeRendering="crispEdges" />
                                </g>
                                <g className="pulse-delayed">
                                <path d="M382.951 639.512L380.01 627.509C379.649 626.034 378.349 625 376.857 625H337.146C335.654 625 334.354 626.034 333.99 627.509L331.049 639.512C331.016 639.641 331 639.777 331 639.913C331 644.48 334.461 648.198 338.719 648.198C341.192 648.198 343.399 646.942 344.812 644.993C346.226 646.942 348.433 648.198 350.906 648.198C353.379 648.198 355.586 646.942 357 644.993C358.414 646.942 360.617 648.198 363.094 648.198C365.57 648.198 367.774 646.942 369.188 644.993C370.601 646.942 372.805 648.198 375.281 648.198C379.539 648.198 383 644.48 383 639.913C383 639.777 382.984 639.641 382.951 639.512Z" fill="white" />
                                <path d="M375.281 651.51C373.068 651.51 370.965 650.82 369.188 649.561C365.632 652.083 360.556 652.083 357 649.561C353.444 652.083 348.368 652.083 344.812 649.561C343.035 650.82 340.932 651.51 338.719 651.51C337.123 651.51 335.618 651.129 334.25 650.479V668.08C334.25 669.909 335.706 671.394 337.5 671.394H350.5V658.138H363.5V671.394H376.5C378.294 671.394 379.75 669.909 379.75 668.08V650.479C378.382 651.129 376.877 651.51 375.281 651.51Z" fill="white" />
                                </g>
                                <g filter="url(#filter2_d_1944_559)">
                                    <circle cx="659.917" cy="647.917" r="44.9167" fill={hexToAdjustedRgba(primaryColor, 1, { r:  0.1125, g: -0.1141304348, b: -0.0390625 })} />
                                    <circle cx="659.917" cy="647.917" r="44.7667" stroke="white" strokeWidth="0.3" />
                                </g>
                                <g filter="url(#filter3_d_1944_559)">
                                    <circle cx="660" cy="648" r="48.85" stroke="white" strokeWidth="0.3" strokeDasharray="2.6 2.6" shapeRendering="crispEdges" />
                                </g>
                                <g className="pulse">
                                <path d="M672.521 655.536C672.123 652.755 670.77 650.212 668.705 648.37C666.641 646.527 664.004 645.507 661.271 645.494H658.724C655.991 645.507 653.354 646.527 651.289 648.37C649.225 650.212 647.871 652.755 647.474 655.536L646.253 664.317C646.214 664.601 646.24 664.89 646.328 665.162C646.417 665.434 646.565 665.68 646.763 665.883C647.241 666.375 650.178 668.887 660 668.887C669.821 668.887 672.751 666.385 673.237 665.883C673.434 665.68 673.583 665.434 673.672 665.162C673.76 664.89 673.786 664.601 673.747 664.317L672.521 655.536Z" fill="white" />
                                <path d="M649.294 646.814C646.952 648.952 645.428 651.825 644.98 654.947L644.084 661.089C636.886 661.041 634.705 658.45 634.341 657.922C634.201 657.731 634.101 657.513 634.046 657.282C633.992 657.051 633.985 656.812 634.026 656.579L634.559 653.604C634.837 652.049 635.49 650.583 636.461 649.331C637.433 648.079 638.695 647.078 640.141 646.411C641.587 645.745 643.174 645.434 644.767 645.504C646.361 645.574 647.914 646.023 649.294 646.814Z" fill="white" />
                                <path d="M685.974 656.579C686.015 656.812 686.008 657.051 685.954 657.282C685.899 657.513 685.799 657.731 685.659 657.922C685.295 658.45 683.114 661.041 675.916 661.089L675.02 654.947C674.572 651.825 673.048 648.952 670.706 646.814C672.086 646.023 673.639 645.574 675.232 645.504C676.826 645.434 678.413 645.745 679.859 646.411C681.305 647.078 682.567 648.079 683.539 649.331C684.51 650.583 685.163 652.049 685.441 653.604L685.974 656.579Z" fill="white" />
                                <path d="M650.059 641.162C649.44 642.023 648.619 642.724 647.665 643.207C646.711 643.689 645.653 643.939 644.58 643.934C643.509 643.934 642.454 643.685 641.501 643.206C640.548 642.727 639.725 642.034 639.099 641.182C638.473 640.33 638.062 639.344 637.901 638.306C637.739 637.268 637.831 636.207 638.17 635.21C638.509 634.214 639.084 633.311 639.848 632.575C640.612 631.84 641.543 631.293 642.565 630.98C643.587 630.667 644.67 630.596 645.725 630.774C646.78 630.952 647.777 631.374 648.634 632.004C648.387 632.949 648.263 633.921 648.265 634.896C648.267 637.106 648.887 639.274 650.059 641.162Z" fill="white" />
                                <path d="M682.176 637.306C682.177 638.177 682.003 639.039 681.663 639.843C681.324 640.648 680.827 641.379 680.199 641.994C679.571 642.61 678.826 643.098 678.006 643.431C677.186 643.764 676.307 643.935 675.42 643.934C674.347 643.939 673.288 643.689 672.334 643.207C671.381 642.724 670.56 642.023 669.941 641.162C671.112 639.274 671.733 637.106 671.735 634.896C671.736 633.921 671.612 632.949 671.366 632.004C672.37 631.265 673.564 630.816 674.813 630.705C676.063 630.594 677.319 630.827 678.442 631.378C679.564 631.928 680.508 632.775 681.167 633.822C681.827 634.869 682.176 636.075 682.176 637.306Z" fill="white" />
                                <path d="M660 643.934C665.068 643.934 669.176 639.92 669.176 634.967C669.176 630.015 665.068 626 660 626C654.932 626 650.823 630.015 650.823 634.967C650.823 639.92 654.932 643.934 660 643.934Z" fill="white" />
                                </g>
                                <g filter="url(#filter4_d_1944_559)">
                                    <circle cx="53.9999" cy="648" r="44.9167" fill={hexToAdjustedRgba(primaryColor, 1, { r:  0.1125, g: -0.1141304348, b: -0.0390625 })} />
                                    <circle cx="53.9999" cy="648" r="44.7667" stroke="white" strokeWidth="0.3" />
                                </g>
                                <g filter="url(#filter5_d_1944_559)">
                                    <circle cx="54" cy="648" r="48.85" stroke="white" strokeWidth="0.3" strokeDasharray="2.6 2.6" shapeRendering="crispEdges" />
                                </g>
                                <g clipPath="url(#clip0_1944_559)" className="pulse">
                                    <path fillRule="evenodd" clipRule="evenodd" d="M56.5164 641.139H68.4209L45.1225 677L50.5647 652.483H39L50.3356 619H67.9108L56.5164 641.139Z" fill="white" />
                                </g>
                                <g filter="url(#filter6_d_1944_559)">
                                    <circle cx="659.917" cy="344.917" r="44.9167" fill={hexToAdjustedRgba(primaryColor, 1, { r:  0.1125, g: -0.1141304348, b: -0.0390625 })} />
                                    <circle cx="659.917" cy="344.917" r="44.7667" stroke="white" strokeWidth="0.3" />
                                </g>
                                <g filter="url(#filter7_d_1944_559)">
                                    <circle cx="660" cy="345" r="48.85" stroke="white" strokeWidth="0.3" strokeDasharray="2.6 2.6" shapeRendering="crispEdges" />
                                </g>
                                <path className="pulse" d="M685.64 332.977C685.348 332.628 684.923 332.427 684.476 332.427H646.305L643.29 321.145C643.109 320.469 642.508 320 641.82 320H635.523C634.681 320 634 320.695 634 321.553C634 322.412 634.681 323.107 635.523 323.107H640.657C652.216 366.291 648.229 351.397 649.092 354.619C647.493 355.351 646.391 356.949 646.391 358.835C646.391 360.984 647.831 362.779 649.773 363.316C649.561 363.853 649.437 364.436 649.437 365.049C649.437 367.619 651.488 369.709 654.008 369.709C656.528 369.709 658.578 367.619 658.578 365.049C658.578 364.502 658.468 363.984 658.297 363.496H671.046C670.875 363.984 670.765 364.502 670.765 365.049C670.765 367.619 672.816 369.709 675.336 369.709C677.856 369.709 679.906 367.619 679.906 365.049C679.906 362.479 677.856 360.389 675.336 360.389H650.961C650.122 360.389 649.437 359.693 649.437 358.835C649.437 357.388 651.266 357.083 650.961 357.252C651.266 357.252 652.296 357.122 681.578 354.168C682.266 354.098 682.819 353.57 682.932 352.877L685.979 334.236C686.053 333.785 685.928 333.325 685.64 332.977ZM655.531 365.049C655.531 365.906 654.847 366.603 654.008 366.603C653.169 366.603 652.484 365.906 652.484 365.049C652.484 364.192 653.169 363.496 654.008 363.496C654.847 363.496 655.531 364.192 655.531 365.049ZM675.336 366.603C674.497 366.603 673.812 365.906 673.812 365.049C673.812 364.192 674.497 363.496 675.336 363.496C676.175 363.496 676.859 364.192 676.859 365.049C676.859 365.906 676.175 366.603 675.336 366.603ZM658.578 346.408C658.578 347.267 657.897 347.962 657.055 347.962C656.213 347.962 655.531 347.267 655.531 346.408V340.194C655.531 339.336 656.213 338.641 657.055 338.641C657.897 338.641 658.578 339.336 658.578 340.194V346.408ZM667.719 346.408C667.719 347.267 667.037 347.962 666.195 347.962C665.353 347.962 664.672 347.267 664.672 346.408V340.194C664.672 339.336 665.353 338.641 666.195 338.641C667.037 338.641 667.719 339.336 667.719 340.194V346.408ZM676.859 346.408C676.859 347.267 676.178 347.962 675.336 347.962C674.494 347.962 673.812 347.267 673.812 346.408V340.194C673.812 339.336 674.494 338.641 675.336 338.641C676.178 338.641 676.859 339.336 676.859 340.194V346.408Z" fill="white" />
                                <path d="M679.906 324.66C679.906 323.802 679.224 323.107 678.382 323.107H675.966L676.413 322.652C677.008 322.045 677.008 321.062 676.413 320.455C675.818 319.848 674.853 319.848 674.258 320.455L672.289 322.464L670.319 320.455C669.724 319.848 668.76 319.848 668.165 320.455C667.569 321.062 667.569 322.045 668.165 322.652L668.611 323.107H666.195C665.353 323.107 664.671 323.802 664.671 324.66V329.321H679.906V324.66Z" fill="white" />
                                <path d="M661.625 326.214H650.96C650.118 326.214 649.437 326.909 649.437 327.767V329.321H661.625V326.214Z" fill="white" />
                                <g filter="url(#filter8_d_1944_559)">
                                    <circle cx="53.9167" cy="344.917" r="44.9167" fill={hexToAdjustedRgba(primaryColor, 1, { r:  0.1125, g: -0.1141304348, b: -0.0390625 })} />
                                    <circle cx="53.9167" cy="344.917" r="44.7667" stroke="white" strokeWidth="0.3" />
                                </g>
                                <g filter="url(#filter9_d_1944_559)">
                                    <circle cx="54" cy="345" r="48.85" stroke="white" strokeWidth="0.3" strokeDasharray="2.6 2.6" shapeRendering="crispEdges" />
                                </g>
                                <path className="pulse-delayed" d="M40.0656 318C33.4109 318 28.0379 323.502 28.0271 330.296L28 347.289C27.9895 353.783 32.9278 359.037 39.1702 359.498V370.115C39.1699 370.29 39.2207 370.461 39.316 370.607C39.4113 370.753 39.5469 370.866 39.7056 370.933C39.8643 371 40.0389 371.017 40.2072 370.983C40.3756 370.948 40.5302 370.864 40.6513 370.74L51.5676 359.586H67.9632C74.6178 359.586 80 354.086 80 347.291V330.297C80 323.504 74.6449 318 67.9902 318H40.0656ZM45.335 336.146C46.7606 336.146 47.9333 337.343 47.9333 338.796C47.9333 340.249 46.7606 341.446 45.335 341.446C43.9094 341.446 42.735 340.249 42.735 338.796C42.735 337.343 43.9094 336.146 45.335 336.146ZM54 336.146C55.4256 336.146 56.6 337.343 56.6 338.796C56.5999 340.249 55.4256 341.446 54 341.446C52.5743 341.446 51.4 340.249 51.4 338.796C51.4 337.343 52.5743 336.146 54 336.146ZM62.6683 336.146C64.094 336.146 65.2683 337.343 65.2683 338.796C65.2683 340.249 64.0939 341.446 62.6683 341.446C61.2427 341.446 60.0666 340.249 60.0666 338.796C60.0666 337.343 61.2427 336.146 62.6683 336.146Z" fill="white" />
                                <g filter="url(#filter10_d_1944_559)">
                                    <circle cx="356.917" cy="51.9167" r="44.9167" fill={hexToAdjustedRgba(primaryColor, 1, { r:  0.1125, g: -0.1141304348, b: -0.0390625 })} />
                                    <circle cx="356.917" cy="51.9167" r="44.7667" stroke="white" strokeWidth="0.3" />
                                </g>
                                <g filter="url(#filter11_d_1944_559)">
                                    <circle cx="357" cy="52" r="48.85" stroke="white" strokeWidth="0.3" strokeDasharray="2.6 2.6" shapeRendering="crispEdges" />
                                </g>
                                <g clipPath="url(#clip1_1944_559)" className="pulse-delayed">
                                    <path d="M341.334 38.6582L340.973 40.4517H350.094L341.334 38.6582Z" fill="white" />
                                    <path d="M336.588 49.7358H339.749C339.782 49.7358 339.825 49.6921 339.825 49.6265V47.2643C339.825 47.1878 339.782 47.144 339.749 47.144H336.588C336.555 47.144 336.512 47.1878 336.512 47.2643V49.6265C336.512 49.6921 336.555 49.7358 336.588 49.7358Z" fill="white" />
                                    <path d="M373.847 49.5938L381.196 51.0045L382.027 47.002L373.847 45.3179V49.5938Z" fill="white" />
                                    <path d="M373.847 69.0817L375.957 69.5192C376.69 69.6832 377.412 69.213 377.576 68.4803C377.576 68.4693 377.576 68.4693 377.576 68.4584L380.933 52.2843L373.847 50.9282V69.0817Z" fill="white" />
                                    <path d="M382.77 41.392C382.573 41.0858 382.267 40.878 381.906 40.8015L343.937 32.9386L343.653 32.9058C343.379 32.9058 343.117 32.9823 342.898 33.1354C342.592 33.3323 342.373 33.6385 342.308 33.9993L341.608 37.3676L356.623 40.4515H371.156C372.644 40.4515 373.847 41.6544 373.847 43.1417V43.9838L382.289 45.7116L382.978 42.4309C383.044 42.07 382.978 41.6982 382.77 41.392Z" fill="white" />
                                    <path d="M371.156 41.7637H332.378C331.612 41.7637 331 42.3761 331 43.1416V69.7157C331.011 70.4703 331.623 71.0827 332.378 71.0936H371.156C371.911 71.1046 372.523 70.4922 372.534 69.7376C372.534 69.7267 372.534 69.7267 372.534 69.7157V43.1416C372.534 42.3761 371.922 41.7637 371.156 41.7637ZM366.684 44.1914H367.996V46.149H366.684V44.1914ZM363.993 44.1914H365.306V46.149H363.993V44.1914ZM360.855 44.1914H362.167V46.149H360.855V44.1914ZM358.165 44.1914H359.477V46.149H358.165V44.1914ZM343.04 63.012H344.998V64.3243H343.04V63.012ZM335.199 49.6266V47.2644C335.188 46.488 335.812 45.8427 336.588 45.8318H339.749C340.525 45.8427 341.148 46.488 341.138 47.2644V49.6266C341.148 50.403 340.536 51.0373 339.76 51.0482H339.749H336.588C335.812 51.0373 335.188 50.4139 335.199 49.6266ZM341.816 63.012V64.3243H339.858V63.012H341.816ZM338.579 63.012V64.3243H336.632V63.012H338.579ZM333.45 63.012H335.396V64.3243H333.45V63.012ZM333.45 68.6221V67.3098H348.235V68.6221H333.45ZM348.235 64.3243H346.277V63.012H348.235V64.3243Z" fill="white" />
                                </g>
                                <g filter="url(#filter12_d_1944_559)">
                                    <circle cx="660" cy="51.9997" r="44.9167" fill={hexToAdjustedRgba(primaryColor, 1, { r:  0.1125, g: -0.1141304348, b: -0.0390625 })} />
                                    <circle cx="660" cy="51.9997" r="44.7667" stroke="white" strokeWidth="0.3" />
                                </g>
                                <g filter="url(#filter13_d_1944_559)">
                                    <circle cx="660" cy="52" r="48.85" stroke="white" strokeWidth="0.3" strokeDasharray="2.6 2.6" shapeRendering="crispEdges" />
                                </g>
                                <path className="pulse" d="M686.44 51.44L663.04 27.59C662.13 26.53 660.83 26 659.4 26H641.2C638.34 26 636 28.385 636 31.3V49.85C636 51.3075 636.52 52.6325 637.56 53.56L660.96 77.41C661.87 78.3375 663.17 79 664.6 79C666.03 79 667.33 78.3375 668.24 77.41L686.44 58.86C687.35 57.9325 688 56.6075 688 55.15C688 53.6925 687.35 52.3675 686.44 51.44ZM645.1 39.25C642.89 39.25 641.2 37.5275 641.2 35.275C641.2 33.0225 642.89 31.3 645.1 31.3C647.31 31.3 649 33.0225 649 35.275C649 37.5275 647.31 39.25 645.1 39.25Z" fill="white" />
                                <g filter="url(#filter14_d_1944_559)">
                                    <circle cx="53.9167" cy="51.9167" r="44.9167" fill={hexToAdjustedRgba(primaryColor, 1, { r:  0.1125, g: -0.1141304348, b: -0.0390625 })} />
                                    <circle cx="53.9167" cy="51.9167" r="44.7667" stroke="white" strokeWidth="0.3" />
                                </g>
                                <g filter="url(#filter15_d_1944_559)">
                                    <circle cx="54" cy="52" r="48.85" stroke="white" strokeWidth="0.3" strokeDasharray="2.6 2.6" shapeRendering="crispEdges" />
                                </g>
                                <path className="pulse" fillRule="evenodd" clipRule="evenodd" d="M45.8298 65.4394L50.856 72.0664C51.3122 72.6679 51.2322 73.5195 50.6781 74.0295L48.9496 75.6209C48.6409 75.905 48.2694 76.0314 47.8542 75.9934C47.4392 75.9553 47.0973 75.7624 46.8425 75.4278L41.2701 68.11L45.8298 65.4394ZM73.4014 48.4317H78.5203C79.3391 48.4317 80.0017 47.7533 80 46.9237C79.9983 46.093 79.3307 45.4218 78.5129 45.4218H73.394C72.5752 45.4218 71.9126 46.1002 71.9143 46.9297C71.9159 47.7606 72.5835 48.4317 73.4014 48.4317ZM67.3253 34.7583L69.8848 30.2608C70.2937 29.5423 70.0547 28.6202 69.3463 28.2033C68.639 27.7871 67.7286 28.0299 67.3193 28.7493L64.7598 33.2467C64.3508 33.9652 64.5899 34.8874 65.2982 35.3042C66.0055 35.7205 66.9159 35.4776 67.3253 34.7583ZM73.2478 40.1454L77.4504 37.6839C78.1638 37.266 78.4082 36.3408 77.9963 35.6169C77.5844 34.8931 76.6724 34.6452 75.9589 35.0631L71.7564 37.5247C71.043 37.9426 70.7986 38.8678 71.2105 39.5917C71.6224 40.3154 72.5344 40.5633 73.2478 40.1454ZM66.6381 46.9186L71.595 55.6288C72.5257 57.2641 71.9682 59.3746 70.3563 60.3188C68.8521 61.1999 66.9402 60.7661 65.935 59.3753H65.9348C60.448 58.5286 53.3007 59.4382 46.1937 62.2488L39.3612 50.243C45.3163 45.4095 49.6798 39.6155 51.7006 34.3715L51.7007 34.3713C50.9908 32.7857 51.5739 30.8663 53.0899 29.9784C54.7019 29.0342 56.7821 29.5998 57.7128 31.2351L62.6697 39.9453C62.8108 39.8347 62.9608 39.7323 63.1195 39.6393C65.0349 38.5174 67.4842 39.1832 68.59 41.1264C69.6958 43.0696 69.0396 45.5545 67.1242 46.6763C66.9655 46.7693 66.8031 46.8499 66.6381 46.9186ZM44.4757 63.5189L37.3163 50.9387L31.6247 54.2724C28.1663 56.2981 26.9705 60.8261 28.9671 64.3346L28.9672 64.3347C30.9639 67.8432 35.4271 69.0564 38.8855 67.0308L44.5771 63.6971L44.4757 63.5189Z" fill="white" />
                                <path d="M260.416 579.093C247.505 579.093 237 568.389 237 555.227V145.096C237 131.933 247.505 121.224 260.416 121.224H438.449C451.36 121.224 461.865 131.933 461.865 145.096V555.221C461.865 568.383 451.36 579.087 438.449 579.087H260.416V579.093Z" fill={hexToAdjustedRgba(primaryColor, 1, {r: 0.2791666667, g: 0.1267605634, b: 0.2204724409})} />
                                <path d="M426.392 421.293C426.392 434.955 415.527 446.032 402.125 446.032H296.468C283.072 446.032 272.201 434.949 272.201 421.293V313.578C272.201 299.916 283.072 288.839 296.468 288.839H402.125C415.533 288.839 426.392 299.916 426.392 313.578V421.293Z" fill="white" />
                                <path d="M426.392 419.629C426.392 433.291 415.527 444.368 402.125 444.368H296.468C283.072 444.368 272.201 433.285 272.201 419.629V311.915C272.201 298.252 283.072 287.175 296.468 287.175H402.125C415.533 287.175 426.392 298.252 426.392 311.915V419.629Z" fill={hexToAdjustedRgba(primaryColor,  1, 0)} />
                                <path fillRule="evenodd" clipRule="evenodd" d="M402.249 341.355C401.256 337.811 398.797 335.895 395.15 335.877C383.906 335.81 372.662 335.762 361.418 335.708C345.723 335.636 330.027 335.569 314.326 335.527C313.463 335.527 313.156 335.232 312.943 334.412C311.737 329.687 310.484 324.975 309.207 320.268C308.113 316.218 305.169 313.837 301.078 313.771C296.787 313.699 292.483 313.765 288.185 313.747C286.311 313.735 285.229 314.789 284.526 316.411C284.526 316.959 284.526 317.507 284.526 318.056C285.306 320.135 286.82 320.87 288.948 320.834C292.784 320.762 296.621 320.81 300.458 320.816C302.012 320.816 302.278 321.033 302.686 322.582C307.753 341.741 312.819 360.893 317.879 380.058C317.986 380.462 318.057 380.878 318.157 381.336C317.814 381.384 317.548 381.426 317.288 381.456C311.359 382.137 307.026 387.923 307.954 393.913C308.811 399.464 313.138 403.255 318.742 403.345C320.303 403.369 321.858 403.351 323.513 403.351C321.379 409.781 322.532 415.187 328.065 418.954C332.339 421.871 337.015 421.859 341.26 418.864C346.675 415.031 347.815 409.661 345.664 403.435C352.858 403.435 360.005 403.435 367.164 403.435C365.397 410.74 366.679 415.386 371.355 418.761C375.369 421.654 380.879 421.708 384.993 418.894C387.057 417.478 388.576 415.585 389.486 413.235C390.793 409.866 390.598 406.521 389.167 403.224C389.38 403.098 389.51 403.02 389.634 402.941C391.112 401.983 391.744 400.5 391.331 398.946C390.923 397.409 389.616 396.396 387.902 396.318C387.559 396.3 387.21 396.318 386.862 396.318C364.528 396.318 342.188 396.318 319.854 396.312C319.274 396.312 318.695 396.324 318.127 396.258C316.449 396.071 315.219 394.908 314.888 393.238C314.35 390.49 316.1 388.441 319.073 388.441C326.829 388.423 334.585 388.447 342.335 388.453C356.606 388.459 370.888 388.477 385.165 388.459C389.652 388.453 392.915 385.765 393.926 381.33C395.859 372.844 397.762 364.353 399.696 355.867C400.588 351.956 401.522 348.051 402.439 344.139C402.439 343.356 402.439 342.572 402.439 341.789C402.379 341.632 402.291 341.506 402.249 341.355ZM378.319 403.351C381.133 403.399 383.421 405.744 383.439 408.618C383.462 411.553 381.157 413.91 378.272 413.916C375.375 413.916 373.082 411.565 373.099 408.624C373.117 405.701 375.488 403.309 378.319 403.351ZM334.538 403.351C337.364 403.327 339.764 405.744 339.77 408.636C339.782 411.547 337.441 413.91 334.562 413.904C331.665 413.904 329.401 411.565 329.413 408.594C329.436 405.726 331.724 403.387 334.538 403.351ZM315.136 342.602C316.502 342.602 317.743 342.602 318.985 342.602C322.549 342.621 326.114 342.633 329.679 342.657C330.187 342.657 330.719 342.524 330.826 343.386C331.399 348.069 332.037 352.751 332.64 357.434C332.682 357.778 332.646 358.127 332.646 358.537C328.224 358.537 323.832 358.537 319.351 358.537C317.962 353.288 316.573 348.032 315.136 342.602ZM335.667 381.408C333.521 381.408 331.423 381.384 329.324 381.414C325.02 381.462 325.387 381.408 324.352 377.479C323.412 373.905 322.461 370.325 321.521 366.751C321.444 366.456 321.391 366.155 321.29 365.66C325.357 365.66 329.259 365.648 333.161 365.691C333.373 365.691 333.746 366.064 333.775 366.299C334.426 371.09 335.041 375.894 335.661 380.685C335.691 380.878 335.667 381.077 335.667 381.408ZM337.701 342.765C342.773 342.765 347.78 342.765 352.87 342.765C352.87 348.002 352.87 353.221 352.87 358.525C348.578 358.525 344.269 358.525 339.77 358.525C339.072 353.282 338.392 348.069 337.701 342.765ZM352.882 381.348C349.5 381.348 346.178 381.348 342.743 381.348C342.063 376.159 341.384 370.988 340.698 365.739C344.806 365.739 348.809 365.739 352.882 365.739C352.882 370.946 352.882 376.117 352.882 381.348ZM371.556 370.247C371.113 373.688 370.67 377.135 370.185 380.57C370.144 380.866 369.818 381.36 369.623 381.36C366.419 381.408 363.209 381.396 359.928 381.396C359.928 376.135 359.928 370.994 359.928 365.709C363.978 365.709 367.95 365.709 372.088 365.709C371.911 367.263 371.746 368.758 371.556 370.247ZM374.53 347.285C374.092 350.726 373.625 354.174 373.194 357.621C373.123 358.218 373.034 358.639 372.272 358.633C368.329 358.591 364.38 358.615 360.431 358.603C360.289 358.603 360.141 358.555 359.94 358.519C359.94 353.3 359.94 348.123 359.94 342.85C364.93 342.85 369.925 342.85 375.068 342.85C374.879 344.38 374.713 345.833 374.53 347.285ZM390.142 366.781C389.179 371.042 388.215 375.303 387.246 379.558C386.927 380.95 386.418 381.39 385.017 381.402C382.416 381.42 379.815 381.408 377.09 381.408C377.356 379.244 377.61 377.207 377.87 375.164C378.231 372.338 378.585 369.511 378.993 366.691C379.046 366.329 379.401 365.715 379.626 365.709C383.143 365.648 386.666 365.666 390.29 365.666C390.231 366.112 390.213 366.456 390.142 366.781ZM392.005 358.555C388.038 358.555 384.142 358.555 380.051 358.555C380.329 356.307 380.583 354.186 380.861 352.064C381.216 349.316 381.565 346.562 381.973 343.82C382.026 343.476 382.422 342.916 382.664 342.916C386.921 342.862 391.171 342.88 395.563 342.88C394.351 348.195 393.181 353.354 392.005 358.555Z" fill="white" />
                                <rect x="276" y="499" width="146.087" height="44.4112" rx="22.2056" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M297.842 529.22C296.639 529.22 295.488 529.051 294.388 528.714C293.288 528.362 292.415 527.915 291.77 527.372L292.76 525.15C293.376 525.634 294.139 526.037 295.048 526.36C295.957 526.683 296.889 526.844 297.842 526.844C298.649 526.844 299.301 526.756 299.8 526.58C300.299 526.404 300.665 526.169 300.9 525.876C301.135 525.568 301.252 525.223 301.252 524.842C301.252 524.373 301.083 523.999 300.746 523.72C300.409 523.427 299.969 523.199 299.426 523.038C298.898 522.862 298.304 522.701 297.644 522.554C296.999 522.407 296.346 522.239 295.686 522.048C295.041 521.843 294.447 521.586 293.904 521.278C293.376 520.955 292.943 520.53 292.606 520.002C292.269 519.474 292.1 518.799 292.1 517.978C292.1 517.142 292.32 516.379 292.76 515.69C293.215 514.986 293.897 514.429 294.806 514.018C295.73 513.593 296.896 513.38 298.304 513.38C299.228 513.38 300.145 513.497 301.054 513.732C301.963 513.967 302.755 514.304 303.43 514.744L302.528 516.966C301.839 516.555 301.127 516.255 300.394 516.064C299.661 515.859 298.957 515.756 298.282 515.756C297.49 515.756 296.845 515.851 296.346 516.042C295.862 516.233 295.503 516.482 295.268 516.79C295.048 517.098 294.938 517.45 294.938 517.846C294.938 518.315 295.099 518.697 295.422 518.99C295.759 519.269 296.192 519.489 296.72 519.65C297.263 519.811 297.864 519.973 298.524 520.134C299.184 520.281 299.837 520.449 300.482 520.64C301.142 520.831 301.736 521.08 302.264 521.388C302.807 521.696 303.239 522.114 303.562 522.642C303.899 523.17 304.068 523.837 304.068 524.644C304.068 525.465 303.841 526.228 303.386 526.932C302.946 527.621 302.264 528.179 301.34 528.604C300.416 529.015 299.25 529.22 297.842 529.22ZM313.306 517.12C314.245 517.12 315.081 517.303 315.814 517.67C316.562 518.037 317.149 518.601 317.574 519.364C317.999 520.112 318.212 521.08 318.212 522.268V529H315.462V522.62C315.462 521.579 315.213 520.801 314.714 520.288C314.23 519.775 313.548 519.518 312.668 519.518C312.023 519.518 311.451 519.65 310.952 519.914C310.453 520.178 310.065 520.574 309.786 521.102C309.522 521.615 309.39 522.268 309.39 523.06V529H306.64V512.676H309.39V520.42L308.796 519.452C309.207 518.704 309.801 518.132 310.578 517.736C311.37 517.325 312.279 517.12 313.306 517.12ZM327.002 529.154C325.799 529.154 324.729 528.897 323.79 528.384C322.851 527.856 322.111 527.137 321.568 526.228C321.025 525.319 320.754 524.285 320.754 523.126C320.754 521.953 321.025 520.919 321.568 520.024C322.111 519.115 322.851 518.403 323.79 517.89C324.729 517.377 325.799 517.12 327.002 517.12C328.219 517.12 329.297 517.377 330.236 517.89C331.189 518.403 331.93 519.107 332.458 520.002C333.001 520.897 333.272 521.938 333.272 523.126C333.272 524.285 333.001 525.319 332.458 526.228C331.93 527.137 331.189 527.856 330.236 528.384C329.297 528.897 328.219 529.154 327.002 529.154ZM327.002 526.8C327.677 526.8 328.278 526.653 328.806 526.36C329.334 526.067 329.745 525.641 330.038 525.084C330.346 524.527 330.5 523.874 330.5 523.126C330.5 522.363 330.346 521.711 330.038 521.168C329.745 520.611 329.334 520.185 328.806 519.892C328.278 519.599 327.684 519.452 327.024 519.452C326.349 519.452 325.748 519.599 325.22 519.892C324.707 520.185 324.296 520.611 323.988 521.168C323.68 521.711 323.526 522.363 323.526 523.126C323.526 523.874 323.68 524.527 323.988 525.084C324.296 525.641 324.707 526.067 325.22 526.36C325.748 526.653 326.342 526.8 327.002 526.8ZM342.414 529.154C341.46 529.154 340.588 528.934 339.796 528.494C339.018 528.054 338.395 527.394 337.926 526.514C337.471 525.619 337.244 524.49 337.244 523.126C337.244 521.747 337.464 520.618 337.904 519.738C338.358 518.858 338.974 518.205 339.752 517.78C340.529 517.34 341.416 517.12 342.414 517.12C343.572 517.12 344.592 517.369 345.472 517.868C346.366 518.367 347.07 519.063 347.584 519.958C348.112 520.853 348.376 521.909 348.376 523.126C348.376 524.343 348.112 525.407 347.584 526.316C347.07 527.211 346.366 527.907 345.472 528.406C344.592 528.905 343.572 529.154 342.414 529.154ZM335.902 533.268V517.252H338.52V520.024L338.432 523.148L338.652 526.272V533.268H335.902ZM342.106 526.8C342.766 526.8 343.352 526.653 343.866 526.36C344.394 526.067 344.812 525.641 345.12 525.084C345.428 524.527 345.582 523.874 345.582 523.126C345.582 522.363 345.428 521.711 345.12 521.168C344.812 520.611 344.394 520.185 343.866 519.892C343.352 519.599 342.766 519.452 342.106 519.452C341.446 519.452 340.852 519.599 340.324 519.892C339.796 520.185 339.378 520.611 339.07 521.168C338.762 521.711 338.608 522.363 338.608 523.126C338.608 523.874 338.762 524.527 339.07 525.084C339.378 525.641 339.796 526.067 340.324 526.36C340.852 526.653 341.446 526.8 342.106 526.8ZM357.328 529V513.6H359.682L369.34 525.458H368.174V513.6H371.012V529H368.658L359 517.142H360.166V529H357.328ZM380.154 529.154C378.952 529.154 377.881 528.897 376.942 528.384C376.004 527.856 375.263 527.137 374.72 526.228C374.178 525.319 373.906 524.285 373.906 523.126C373.906 521.953 374.178 520.919 374.72 520.024C375.263 519.115 376.004 518.403 376.942 517.89C377.881 517.377 378.952 517.12 380.154 517.12C381.372 517.12 382.45 517.377 383.388 517.89C384.342 518.403 385.082 519.107 385.61 520.002C386.153 520.897 386.424 521.938 386.424 523.126C386.424 524.285 386.153 525.319 385.61 526.228C385.082 527.137 384.342 527.856 383.388 528.384C382.45 528.897 381.372 529.154 380.154 529.154ZM380.154 526.8C380.829 526.8 381.43 526.653 381.958 526.36C382.486 526.067 382.897 525.641 383.19 525.084C383.498 524.527 383.652 523.874 383.652 523.126C383.652 522.363 383.498 521.711 383.19 521.168C382.897 520.611 382.486 520.185 381.958 519.892C381.43 519.599 380.836 519.452 380.176 519.452C379.502 519.452 378.9 519.599 378.372 519.892C377.859 520.185 377.448 520.611 377.14 521.168C376.832 521.711 376.678 522.363 376.678 523.126C376.678 523.874 376.832 524.527 377.14 525.084C377.448 525.641 377.859 526.067 378.372 526.36C378.9 526.653 379.494 526.8 380.154 526.8ZM391.328 529L387.016 517.252H389.612L393.308 527.614H392.076L395.948 517.252H398.258L402.02 527.614H400.81L404.616 517.252H407.058L402.724 529H400.084L396.652 519.826H397.466L393.968 529H391.328Z" fill="white" />
                                <path d="M231.57 183.281C230.618 183.281 229.85 184.064 229.85 185.028V194.605C229.85 195.575 230.618 196.359 231.57 196.359H233.887V183.281H231.57Z" fill="#161616" />
                                <path d="M231.57 214.162C230.618 214.162 229.85 214.945 229.85 215.91V248.013C229.85 248.978 230.618 249.761 231.57 249.761H233.887V214.156H231.57V214.162Z" fill="#161616" />
                                <path d="M467.028 225.305C467.98 225.305 468.749 226.088 468.749 227.052V259.156C468.749 260.127 467.98 260.91 467.028 260.91H464.705V225.305H467.028Z" fill="#161616" />
                                <path d="M231.57 257.276C230.618 257.276 229.85 258.06 229.85 259.024V291.128C229.85 292.098 230.618 292.882 231.57 292.882H233.887V257.276H231.57Z" fill="#161616" />
                                <path d="M438.316 117H260.282C245.119 117 232.723 129.644 232.723 145.096V555.222C232.723 570.674 245.125 583.318 260.282 583.318H438.316C453.473 583.318 465.876 570.674 465.876 555.222V145.096C465.876 129.644 453.473 117 438.316 117ZM458.267 555.228C458.267 566.443 449.317 575.568 438.31 575.568H260.282C249.275 575.568 240.325 566.443 240.325 555.228V145.096C240.325 133.875 249.275 124.75 260.282 124.75H338.57C340.585 124.75 341.839 126.619 342.607 128.68C343.376 130.735 344.144 134.116 349.299 134.116C354.448 134.116 355.217 130.735 355.985 128.68C356.754 126.625 358.001 124.75 360.023 124.75H438.322C449.323 124.75 458.279 133.875 458.279 145.096V555.228H458.267Z" fill="url(#paint0_linear_1944_559)" />
                                <path d="M352.078 128.577C352.078 130.144 350.83 131.415 349.293 131.415C347.75 131.415 346.509 130.144 346.509 128.577C346.509 127.01 347.756 125.738 349.293 125.738C350.836 125.738 352.078 127.01 352.078 128.577Z" fill="#1D1B21" />
                                <path d="M350.736 128.577C350.736 129.391 350.091 130.047 349.299 130.047C348.501 130.047 347.857 129.391 347.857 128.577C347.857 127.769 348.501 127.106 349.299 127.106C350.091 127.112 350.736 127.769 350.736 128.577Z" fill="#121114" />
                                <path d="M458.912 230.819C458.817 219.706 458.569 210.443 458.474 203.964C458.356 197.48 458.285 193.779 458.285 193.779V430.837C458.285 430.837 458.356 427.131 458.474 420.652C458.569 414.168 458.817 404.911 458.912 393.798C459 382.685 459.213 369.721 459.249 355.83C459.296 341.938 459.343 327.125 459.39 312.311C459.343 297.492 459.29 282.678 459.243 268.787C459.213 254.896 459.006 241.932 458.912 230.819Z" fill="url(#paint1_linear_1944_559)" />
                                <path d="M239.681 505.17C239.781 516.283 240.023 525.546 240.118 532.025C240.236 538.503 240.307 542.21 240.307 542.21V305.151C240.307 305.151 240.236 308.858 240.118 315.336C240.023 321.821 239.775 331.078 239.681 342.191C239.586 353.304 239.379 366.268 239.344 380.159C239.296 394.05 239.249 408.864 239.196 423.677C239.243 438.491 239.296 453.31 239.344 467.202C239.379 481.093 239.586 494.057 239.681 505.17Z" fill="url(#paint2_linear_1944_559)" />
                                <path d="M232.747 206.273C232.747 213.565 226.947 219.471 219.794 219.471C212.641 219.471 206.842 213.559 206.842 206.273V196.907H232.741V206.273H232.747Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M258.645 206.273C258.645 213.565 252.846 219.471 245.699 219.471C238.546 219.471 232.746 213.559 232.746 206.273V196.907H258.645V206.273Z" fill="url(#paint3_linear_1944_559)" />
                                <path d="M284.544 206.273C284.544 213.565 278.744 219.471 271.591 219.471C264.438 219.471 258.639 213.559 258.639 206.273V196.907H284.538V206.273H284.544Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M310.448 206.273C310.448 213.565 304.649 219.471 297.496 219.471C290.337 219.471 284.544 213.559 284.544 206.273V196.907H310.442V206.273H310.448Z" fill="url(#paint4_linear_1944_559)" />
                                <path d="M336.347 206.273C336.347 213.565 330.548 219.471 323.395 219.471C316.242 219.471 310.442 213.559 310.442 206.273V196.907H336.341V206.273H336.347Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M362.246 206.273C362.246 213.565 356.446 219.471 349.293 219.471C342.14 219.471 336.341 213.559 336.341 206.273V196.907H362.24V206.273H362.246Z" fill="url(#paint5_linear_1944_559)" />
                                <path d="M388.144 206.273C388.144 213.565 382.345 219.471 375.192 219.471C368.039 219.471 362.24 213.559 362.24 206.273V196.907H388.138V206.273H388.144Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M414.043 206.273C414.043 213.565 408.249 219.471 401.096 219.471C393.943 219.471 388.144 213.559 388.144 206.273V196.907H414.043V206.273Z" fill="url(#paint6_linear_1944_559)" />
                                <path d="M439.942 206.273C439.942 213.565 434.142 219.471 426.989 219.471C419.836 219.471 414.037 213.559 414.037 206.273V196.907H439.936V206.273H439.942Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M465.846 206.273C465.846 213.565 460.047 219.471 452.894 219.471C445.741 219.471 439.942 213.559 439.942 206.273V196.907H465.84V206.273H465.846Z" fill="url(#paint7_linear_1944_559)" />
                                <path d="M491.745 206.273C491.745 213.565 485.946 219.471 478.793 219.471C471.64 219.471 465.84 213.559 465.84 206.273V196.907H491.739V206.273H491.745Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M259.898 162.29H240.035L206.83 196.901H232.723L259.898 162.29Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M279.761 162.29H259.898L232.723 196.901H258.621L279.761 162.29Z" fill="url(#paint8_linear_1944_559)" />
                                <path d="M299.624 162.29H279.761L258.621 196.901H284.526L299.624 162.29Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M319.487 162.29H299.624L284.526 196.901H310.425L319.487 162.29Z" fill="url(#paint9_linear_1944_559)" />
                                <path d="M339.35 162.29H319.487L310.425 196.901H336.323L339.35 162.29Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M359.219 162.29H339.35L336.323 196.901H362.228L359.219 162.29Z" fill="url(#paint10_linear_1944_559)" />
                                <path d="M379.082 162.29H359.219L362.228 196.901H388.126L379.082 162.29Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M398.939 162.29H379.082L388.127 196.901H414.019L398.939 162.29Z" fill="url(#paint11_linear_1944_559)" />
                                <path d="M418.808 162.29H398.939L414.019 196.901H439.924L418.808 162.29Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M438.671 162.29H418.808L439.924 196.901H465.823L438.671 162.29Z" fill="url(#paint12_linear_1944_559)" />
                                <path d="M458.534 162.29H438.671L465.823 196.901H491.721L458.534 162.29Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M259.898 162.29V124.774C249.068 124.985 240.325 134.007 240.325 145.096V162.29H259.898Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M279.761 124.69H259.898V162.29H279.761V124.69Z" fill="url(#paint13_linear_1944_559)" />
                                <path d="M299.624 124.69H279.761V162.29H299.624V124.69Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M319.487 124.69H299.624V162.29H319.487V124.69Z" fill="url(#paint14_linear_1944_559)" />
                                <path d="M339.356 124.69H319.493V162.29H339.356V124.69Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M359.219 124.69H339.356V162.29H359.219V124.69Z" fill="url(#paint15_linear_1944_559)" />
                                <path d="M379.082 124.69H359.219V162.29H379.082V124.69Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M398.945 124.69H379.082V162.29H398.945V124.69Z" fill="url(#paint16_linear_1944_559)" />
                                <path d="M418.808 124.69H398.945V162.29H418.808V124.69Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path d="M438.676 124.69H418.813V162.29H438.676V124.69Z" fill="url(#paint17_linear_1944_559)" />
                                <path d="M438.676 124.768V162.284H458.267V145.096C458.267 134.001 449.512 124.973 438.676 124.768Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <path opacity="0.5" d="M491.722 196.907H206.848V197.516H491.722V196.907Z" fill="white" />
                                <path opacity="0.5" d="M458.267 161.519H240.035V162.29H458.267V161.519Z" fill="white" />
                                <g clipPath="url(#clip2_1944_559)">
                                    <path d="M526.482 233.979L500.818 263.187L549.147 263.08L526.482 233.979Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                    <path d="M597 217.387C597 242.679 576.973 263.187 552.274 263.187C527.575 263.187 507.548 242.679 507.548 217.387C507.548 192.096 527.575 171.588 552.274 171.588C576.973 171.588 597 192.096 597 217.387Z" fill={hexToAdjustedRgba(primaryColor, 1, { r: 0.121, g: -0.201, b: -0.117 })} />
                                <g className="pulse-large">
                                    <path d="M573.835 221.28C573.835 219.703 572.812 218.381 571.417 217.923C573.044 217.501 574.259 216.001 574.259 214.21C574.259 212.097 572.574 210.365 570.511 210.365H559.381C559.765 210.044 560.073 209.622 560.265 209.134L560.91 207.443C562.026 204.51 562.543 201.379 562.427 198.237C562.363 196.613 561.061 195.327 559.474 195.327C558.161 195.327 557.004 196.214 556.638 197.505L555.447 201.683C555.342 202.046 555.174 202.391 554.947 202.695L547.572 212.401C547.532 212.448 547.479 212.49 547.433 212.538C545.945 214.251 545.172 216.65 545.172 219.113V234.509C545.172 235.592 546.044 236.484 547.102 236.484H569.279C571.01 236.484 572.428 235.032 572.428 233.253C572.428 231.896 571.597 230.73 570.423 230.254C571.882 229.903 572.98 228.564 572.98 226.969C572.98 225.969 572.539 225.076 571.859 224.457C573.027 223.886 573.835 222.678 573.835 221.28Z" fill="white" />
                                    <path d="M540.947 214.418H531.701C530.916 214.418 530.283 215.073 530.283 215.87V235.026C530.283 235.83 530.922 236.484 531.701 236.484H540.947C541.726 236.484 542.365 235.83 542.365 235.026V215.87C542.365 215.067 541.726 214.418 540.947 214.418Z" fill="white" />
                                </g>
                                </g>
                                <rect width="22.1926" height="22.6295" rx="11.0963" transform="matrix(-1 0 0 1 176.395 225.252)" fill="white" fillOpacity="0.1" />
                                <rect x="117.907" y="170.377" width="9.86338" height="10.0575" rx="4.93169" fill="white" fillOpacity="0.1" />
                                <rect width="9.86338" height="10.0575" rx="4.93169" transform="matrix(-1 0 0 1 178.768 121.626)" fill="white" fillOpacity="0.1" />
                                <rect width="22.1926" height="22.6295" rx="11.0963" transform="matrix(-1 0 0 1 574.731 391.794)" fill="white" fillOpacity="0.1" />
                                <rect width="22.1926" height="22.6295" rx="11.0963" transform="matrix(-1 0 0 1 541.193 116)" fill="white" fillOpacity="0.1" />
                                <rect x="491.745" y="357.274" width="9.86338" height="10.0575" rx="4.93169" fill="white" fillOpacity="0.1" />
                                <rect width="9.86338" height="10.0575" rx="4.93169" transform="matrix(-1 0 0 1 552.606 308.523)" fill="white" fillOpacity="0.1" />
                                <rect className="pulse-large" x="113" y="414" width="18.9534" height="19.3265" rx="9.47671" fill="white" fillOpacity="0.1" />
                                <rect className="pulse-large" x="496.282" y="468.589" width="18.9534" height="19.3265" rx="9.47671" fill="white" fillOpacity="0.1" />
                                <rect x="207.737" y="488.019" width="8.42374" height="8.58956" rx="4.21187" fill="white" fillOpacity="0.1" />
                                <rect x="577.038" y="496.346" width="8.42374" height="8.58956" rx="4.21187" fill="white" fillOpacity="0.1" />
                                <rect x="153.878" y="548.421" width="18.1474" height="18.5047" rx="9.07372" fill="white" fillOpacity="0.1" />
                                <rect className="pulse-large" x="534.392" y="556.486" width="18.1474" height="18.5047" rx="9.07372" fill="white" fillOpacity="0.1" />
                                <defs>
                                    <filter id="filter0_d_1944_559" x="307" y="600" width="99.8333" height="99.8335" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter1_d_1944_559" x="303" y="596" width="108" height="108" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter2_d_1944_559" x="610" y="600" width="99.8333" height="99.8335" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter3_d_1944_559" x="606" y="596" width="108" height="108" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter4_d_1944_559" x="4.08325" y="600.083" width="99.8334" height="99.8335" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter5_d_1944_559" x="0" y="596" width="108" height="108" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter6_d_1944_559" x="610" y="297" width="99.8333" height="99.8335" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter7_d_1944_559" x="606" y="293" width="108" height="108" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter8_d_1944_559" x="4" y="297" width="99.8334" height="99.8335" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter9_d_1944_559" x="0" y="293" width="108" height="108" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter10_d_1944_559" x="307" y="4" width="99.8333" height="99.8335" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter11_d_1944_559" x="303" y="0" width="108" height="108" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter12_d_1944_559" x="610.083" y="4.08301" width="99.8333" height="99.8335" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter13_d_1944_559" x="606" y="0" width="108" height="108" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter14_d_1944_559" x="4" y="4" width="99.8334" height="99.8335" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <filter id="filter15_d_1944_559" x="0" y="0" width="108" height="108" filterUnits="userSpaceOnUse" colorInterpolationFilters="sRGB">
                                        <feFlood floodOpacity="0" result="BackgroundImageFix" />
                                        <feColorMatrix in="SourceAlpha" type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0" result="hardAlpha" />
                                        <feOffset dy="2" />
                                        <feGaussianBlur stdDeviation="2.5" />
                                        <feComposite in2="hardAlpha" operator="out" />
                                        <feColorMatrix type="matrix" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0.2 0" />
                                        <feBlend mode="normal" in2="BackgroundImageFix" result="effect1_dropShadow_1944_559" />
                                        <feBlend mode="normal" in="SourceGraphic" in2="effect1_dropShadow_1944_559" result="shape" />
                                    </filter>
                                    <linearGradient id="paint0_linear_1944_559" x1="232.72" y1="350.16" x2="465.875" y2="350.16" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="#282433" />
                                        <stop offset="0.1024" stopColor="#322E3D" />
                                        <stop offset="0.2581" stopColor="#494654" />
                                        <stop offset="0.2809" stopColor="#474452" />
                                        <stop offset="0.4987" stopColor="#363241" />
                                        <stop offset="0.7308" stopColor="#2B2736" />
                                        <stop offset="1" stopColor="#282433" />
                                    </linearGradient>
                                    <linearGradient id="paint1_linear_1944_559" x1="458.846" y1="239.257" x2="458.846" y2="387.548" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="#282433" />
                                        <stop offset="0.1024" stopColor="#322E3D" />
                                        <stop offset="0.2581" stopColor="#494654" />
                                        <stop offset="0.2809" stopColor="#474452" />
                                        <stop offset="0.4987" stopColor="#363241" />
                                        <stop offset="0.7308" stopColor="#2B2736" />
                                        <stop offset="1" stopColor="#282433" />
                                    </linearGradient>
                                    <linearGradient id="paint2_linear_1944_559" x1="239.746" y1="496.729" x2="239.746" y2="348.438" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="#282433" />
                                        <stop offset="0.1024" stopColor="#322E3D" />
                                        <stop offset="0.2581" stopColor="#494654" />
                                        <stop offset="0.2809" stopColor="#474452" />
                                        <stop offset="0.4987" stopColor="#363241" />
                                        <stop offset="0.7308" stopColor="#2B2736" />
                                        <stop offset="1" stopColor="#282433" />
                                    </linearGradient>
                                    <linearGradient id="paint3_linear_1944_559" x1="245.696" y1="203.998" x2="245.696" y2="265.669" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint4_linear_1944_559" x1="297.496" y1="203.998" x2="297.496" y2="265.669" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint5_linear_1944_559" x1="349.295" y1="203.998" x2="349.295" y2="265.669" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint6_linear_1944_559" x1="401.094" y1="203.998" x2="401.094" y2="265.669" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint7_linear_1944_559" x1="452.894" y1="203.998" x2="452.894" y2="265.669" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint8_linear_1944_559" x1="256.242" y1="165.934" x2="256.242" y2="209.173" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint9_linear_1944_559" x1="302.008" y1="165.934" x2="302.008" y2="209.173" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint10_linear_1944_559" x1="349.273" y1="165.934" x2="349.273" y2="209.173" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint11_linear_1944_559" x1="396.549" y1="165.934" x2="396.549" y2="209.173" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint12_linear_1944_559" x1="442.315" y1="165.934" x2="442.315" y2="209.173" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint13_linear_1944_559" x1="269.83" y1="131.403" x2="269.83" y2="162.799" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint14_linear_1944_559" x1="309.558" y1="131.403" x2="309.558" y2="162.799" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint15_linear_1944_559" x1="349.287" y1="131.403" x2="349.287" y2="162.799" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint16_linear_1944_559" x1="389.014" y1="131.403" x2="389.014" y2="162.799" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <linearGradient id="paint17_linear_1944_559" x1="428.743" y1="131.403" x2="428.743" y2="162.799" gradientUnits="userSpaceOnUse">
                                        <stop stopColor="white" />
                                        <stop offset="0.0663" stopColor="#FAFAFC" />
                                        <stop offset="0.349" stopColor="#E8E9F1" />
                                        <stop offset="0.6504" stopColor="#DDDFEA" />
                                        <stop offset="1" stopColor="#DADCE8" />
                                    </linearGradient>
                                    <clipPath id="clip0_1944_559">
                                        <rect width="52" height="52" fill="white" transform="translate(28 622)" />
                                    </clipPath>
                                    <clipPath id="clip1_1944_559">
                                        <rect width="52" height="52" fill="white" transform="translate(331 26)" />
                                    </clipPath>
                                    <clipPath id="clip2_1944_559">
                                        <rect width="96.1814" height="91.5981" fill="white" transform="translate(500.818 171.588)" />
                                    </clipPath>
                                </defs>
                            </svg>
                        </div>
                    </div>
                </div>
            </div>
            
            {/* Right side - Content */}
            <div className="w-full lg:w-1/2 flex items-center justify-center p-6 md:p-12 relative bg-slate-50 dark:bg-slate-900">
                {/* Background decoration */}
                <div className="absolute inset-0 overflow-hidden">
                    <div 
                        className="absolute -top-40 -right-40 w-80 h-80 rounded-full opacity-20"
                        style={{ backgroundColor: `${primaryColor}20` }}
                    ></div>
                    <div 
                        className="absolute -bottom-32 -left-32 w-64 h-64 rounded-full opacity-15"
                        style={{ backgroundColor: `${primaryColor}30` }}
                    ></div>
                </div>
                {/* Language Switcher - Top Right */}
                <div className="absolute top-4 right-4">
                    <LanguageSwitcher />
                </div>
                
                <div 
                    className={`w-full max-w-md transition-all duration-700 ${
                        mounted ? 'opacity-100 translate-y-0' : 'opacity-0 translate-y-4'
                    }`}
                >
                    {/* Mobile branding - only visible on small screens */}
                    <div className="flex flex-col items-center mb-8 lg:hidden">
                        <div 
                            className="p-4 rounded-xl shadow-lg inline-flex mb-4"
                            style={{ backgroundColor: primaryColor }}
                        >
                            {currentLogo ? (
                                <img src={currentLogo} alt="Logo" className="h-8 w-8 object-contain" />
                            ) : (
                                <CreditCard className="h-8 w-8 text-white" />
                            )}
                        </div>
                    </div>
                    
                    <div className="bg-white dark:bg-slate-800 rounded-2xl shadow-xl p-8 border border-slate-200 dark:border-slate-700">
                        <div className="text-center mb-6">
                            {icon && (
                                <div 
                                    className="mx-auto mb-4 flex h-14 w-14 items-center justify-center rounded-full"
                                    style={{ backgroundColor: `${primaryColor}20` }}
                                >
                                    {icon}
                                </div>
                            )}
                            <h1 className="text-2xl font-bold text-slate-900 dark:text-white mb-2">{title}</h1>
                            {description && (
                                <p className="text-slate-600 dark:text-slate-400">{description}</p>
                            )}
                        </div>
                        
                        {status && (
                            <div className={`mb-6 text-center text-sm font-medium ${
                                statusType === 'success' 
                                    ? 'text-green-700 dark:text-green-400 bg-green-50 dark:bg-green-900/20 border-green-200 dark:border-green-800/30' 
                                    : 'text-red-700 dark:text-red-400 bg-red-50 dark:bg-red-900/20 border-red-200 dark:border-red-800/30'
                            } p-3 rounded-lg border`}>
                                {status}
                            </div>
                        )}
                        
                        {children}
                    </div>
                    

                </div>
            </div>
        </div>
    );
}
