import React from 'react';
import { Link } from '@inertiajs/react';
import { ArrowRight, Play } from 'lucide-react';
import { getImageUrl } from '@/utils/image-helper';

interface HeroSectionProps {
  brandColor?: string;
  settings: any;
  sectionData: {
    title?: string;
    subtitle?: string;
    announcement_text?: string;
    primary_button_text?: string;
    secondary_button_text?: string;
    stats?: Array<{value: string; label: string}>;
    image?: string;
  };
}

export default function HeroSection({ settings, sectionData, brandColor = '#3b82f6' }: HeroSectionProps) {
  // Get colors from settings
  const configColors = settings.config_sections?.colors;
  const secondaryColor = configColors?.secondary || '#059669';

  return (
    <section id="hero" className="relative pt-16 bg-gradient-to-br from-gray-50 to-white min-h-screen flex items-center overflow-hidden">
      {/* Background decoration */}
      <div className="absolute inset-0 bg-grid-gray-100 bg-[size:20px_20px] opacity-50"></div>
      <div className="absolute top-0 right-0 w-96 h-96 bg-gradient-to-br from-blue-100 to-purple-100 rounded-full blur-3xl opacity-30 -translate-y-1/2 translate-x-1/2"></div>
      <div className="absolute bottom-0 left-0 w-96 h-96 bg-gradient-to-tr from-green-100 to-blue-100 rounded-full blur-3xl opacity-30 translate-y-1/2 -translate-x-1/2"></div>
      
      <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12 sm:py-16 lg:py-20">
        <div className="grid lg:grid-cols-2 gap-8 sm:gap-12 lg:gap-16 items-center">
          {/* Left Content */}
          <div className="text-center lg:text-left space-y-6 sm:space-y-8">
            {sectionData.announcement_text && (
              <div className="inline-flex items-center px-4 py-2 bg-gray-100 text-gray-700 rounded-full text-sm font-medium border">
                {sectionData.announcement_text}
              </div>
            )}
            
            <h1 className="text-4xl md:text-5xl lg:text-6xl font-bold text-gray-900 leading-tight" role="banner" aria-label="Main heading">
              {sectionData.title || 'Launch Your Online Store in Minutes'}
            </h1>
            
            <p className="text-lg md:text-xl text-gray-600 leading-relaxed max-w-2xl font-medium">
              {sectionData.subtitle || 'Create, customize, and manage multiple online stores with our powerful e-commerce platform.'}
            </p>
            
            <div className="flex flex-col sm:flex-row gap-3 sm:gap-4 justify-center lg:justify-start">
              <Link
                href={route('register')}
                className="group text-white px-8 py-4 rounded-xl transition-all duration-300 font-semibold text-base flex items-center justify-center gap-2 shadow-lg hover:shadow-xl transform hover:-translate-y-0.5"
                style={{ backgroundColor: brandColor }}
                onMouseEnter={(e) => {
                  e.currentTarget.style.backgroundColor = `${brandColor}dd`;
                }}
                onMouseLeave={(e) => {
                  e.currentTarget.style.backgroundColor = brandColor;
                }}
                aria-label="Start free trial - Register for WhatsStore"
              >
                {sectionData.primary_button_text || 'Start Free Trial'}
                <ArrowRight size={18} className="group-hover:translate-x-1 transition-transform" />
              </Link>
              <Link
                href={route('login')}
                className="group px-8 py-4 rounded-xl transition-all duration-300 font-semibold text-base flex items-center justify-center gap-2 shadow-sm hover:shadow-md text-white"
                style={{ backgroundColor: secondaryColor }}
                onMouseEnter={(e) => {
                  e.currentTarget.style.backgroundColor = `${secondaryColor}dd`;
                }}
                onMouseLeave={(e) => {
                  e.currentTarget.style.backgroundColor = secondaryColor;
                }}
                aria-label="Login to existing WhatsStore account"
              >
                <Play size={18} className="group-hover:scale-110 transition-transform" />
                {sectionData.secondary_button_text || 'Login'}
              </Link>
            </div>

            {sectionData.stats && sectionData.stats.length > 0 && (
              <div className="grid grid-cols-3 gap-4 sm:gap-6 lg:gap-8 pt-8 sm:pt-12">
                {sectionData.stats.map((stat, index) => (
                  <div key={index} className="text-center">
                    <div className="text-3xl md:text-4xl font-bold text-gray-900">
                      {stat.value}
                    </div>
                    <div className="text-gray-600 font-medium text-sm">{stat.label}</div>
                  </div>
                ))}
              </div>
            )}
          </div>

          {/* Right Content - Hero Image */}
          <div className="relative lg:flex lg:justify-center">
            {/* Floating elements */}
            <div className="absolute -top-8 -right-8 w-20 h-20 bg-gradient-to-br from-blue-200 to-purple-200 rounded-full opacity-60 animate-pulse"></div>
            <div className="absolute -bottom-8 -left-8 w-16 h-16 bg-gradient-to-br from-green-200 to-blue-200 rounded-full opacity-50 animate-pulse" style={{ animationDelay: '1s' }}></div>
            
            <div className="relative transform hover:scale-105 transition-transform duration-300">
              <div className="bg-white rounded-3xl shadow-2xl p-6 max-w-lg mx-auto border border-gray-100 backdrop-blur-sm">
                <img 
                  src={getImageUrl(sectionData.image || '/images/hero-dashboard.svg')}
                  alt="WhatsStore Dashboard Preview"
                  className="w-full h-auto rounded-2xl"
                  onError={(e) => {
                    e.currentTarget.src = getImageUrl('/images/hero-dashboard.svg');
                  }}
                />
              </div>
            </div>
          </div>
        </div>
      </div>
    </section>
  );
}